#!/opt/support/venv/bin/python3
"""Wrapper around pkgacct for tier2c"""
import argparse
import sys
import psutil
import rads

sys.path.insert(0, '/opt/support/lib')
from arg_types import cpuser_safe_arg
from run_cmd import cpuwatch_execv

# This tool will ensure at least this much space is free after pkgacct
DISK_PERCENT_LIMIT = 1.5


def check_acct_size(user: str):
    """Ensure the account is not too large to pkgacct in full"""
    disk_usage = psutil.disk_usage('/home')
    gb_free = int(disk_usage.free / 2**30)
    gb_total = int(disk_usage.total / 2**30)
    try:
        acct_gb = int(rads.QuotaCtl().getquota(user=user) / 2**30)
    except rads.QuotasDisabled as exc:
        sys.exit(f"{exc} - try /scripts/fixquotas")
    disk_needed = gb_total * DISK_PERCENT_LIMIT / 100
    if gb_free - acct_gb > disk_needed:
        return
    print(f"Cannot package {user} without --skiphomedir")
    print(f"The account would place the disk at < {DISK_PERCENT_LIMIT}% free")
    print(f"Account size: {acct_gb} GB")
    print(f"Disk free space: {gb_free} GB")
    sys.exit(1)


def parse_args():
    """Parse commandline arguments"""
    parser = argparse.ArgumentParser(
        description='Wrapper around CPanel pkgacct. It is recommended to run '
        'this in screen due to the amount of time it may take.'
    )
    parser.add_argument(
        '--skiphomedir',
        action='store_true',
        help='Do not include the home directory in the CPanel package',
    )
    parser.add_argument(
        'user',
        metavar='USER',
        type=cpuser_safe_arg,
        help='user to create a CPanel package for',
    )
    args = parser.parse_args()
    return args


def main():
    """main: parse args, determine size, run pkgacct"""
    args = parse_args()
    if not args.skiphomedir:
        check_acct_size(user=args.user)
    cmd = ['/usr/local/cpanel/scripts/pkgacct']
    if args.skiphomedir:
        cmd.append('--skiphomedir')
    cmd.append(args.user)
    cpuwatch_execv(cmd)


if __name__ == '__main__':
    main()
